'use client';

import { EventFilters, EventType } from '@/types/event';
import { Search, Filter, X, ChevronDown } from 'lucide-react';
import { motion, AnimatePresence } from 'framer-motion';
import { useState } from 'react';

interface SearchFiltersProps {
  filters: EventFilters;
  onFilterChange: (filters: EventFilters) => void;
  countries: string[];
  eventTypes: EventType[];
  totalResults: number;
}

/**
 * Format event type for display
 */
function formatEventType(type: string): string {
  return type.split('-').map(word => 
    word.charAt(0).toUpperCase() + word.slice(1)
  ).join(' ');
}

/**
 * SearchFilters component - Light Theme
 */
export default function SearchFilters({
  filters,
  onFilterChange,
  countries,
  eventTypes,
  totalResults,
}: SearchFiltersProps) {
  const [showFilters, setShowFilters] = useState(false);

  const updateFilter = <K extends keyof EventFilters>(key: K, value: EventFilters[K]) => {
    onFilterChange({ ...filters, [key]: value });
  };

  const clearFilters = () => {
    onFilterChange({
      search: '',
      eventType: 'all',
      country: 'all',
      dateRange: 'all',
      ticketAvailable: null,
    });
  };

  const hasActiveFilters = 
    filters.eventType !== 'all' || 
    filters.country !== 'all' || 
    filters.dateRange !== 'all' ||
    filters.ticketAvailable !== null;

  return (
    <div className="space-y-4">
      {/* Search Bar */}
      <div className="relative">
        <div className="absolute inset-y-0 left-0 pl-4 flex items-center pointer-events-none">
          <Search className="w-5 h-5 text-gray-400" />
        </div>
        <input
          type="text"
          placeholder="Search events, venues, or locations..."
          value={filters.search}
          onChange={(e) => updateFilter('search', e.target.value)}
          className="w-full pl-12 pr-12 py-4 bg-white border border-gray-200 rounded-xl 
                     text-gray-900 placeholder-gray-400 focus:outline-none focus:border-[#E11D2E] 
                     focus:ring-2 focus:ring-[#E11D2E]/10 transition-all text-lg shadow-sm"
        />
        {filters.search && (
          <button
            onClick={() => updateFilter('search', '')}
            className="absolute inset-y-0 right-4 flex items-center text-gray-400 hover:text-gray-600 transition-colors"
          >
            <X className="w-5 h-5" />
          </button>
        )}
      </div>

      {/* Filter Toggle & Active Filters */}
      <div className="flex flex-wrap items-center gap-3">
        <button
          onClick={() => setShowFilters(!showFilters)}
          className={`
            flex items-center gap-2 px-4 py-2 rounded-lg border transition-all
            ${showFilters 
              ? 'bg-[#FEE2E2] border-[#E11D2E] text-[#E11D2E]' 
              : 'bg-white border-gray-200 text-gray-600 hover:border-[#E11D2E]/50'
            }
          `}
        >
          <Filter className="w-4 h-4" />
          <span className="text-sm font-medium">Filters</span>
          {hasActiveFilters && (
            <span className="w-2 h-2 rounded-full bg-[#E11D2E]" />
          )}
          <ChevronDown className={`w-4 h-4 transition-transform ${showFilters ? 'rotate-180' : ''}`} />
        </button>

        {/* Results count */}
        <span className="text-sm text-gray-500">
          {totalResults} event{totalResults !== 1 ? 's' : ''} found
        </span>

        {/* Clear filters button */}
        {hasActiveFilters && (
          <button
            onClick={clearFilters}
            className="flex items-center gap-1 px-3 py-1.5 rounded-lg text-sm text-[#E11D2E] 
                       hover:bg-[#FEE2E2] transition-colors ml-auto"
          >
            <X className="w-4 h-4" />
            Clear Filters
          </button>
        )}
      </div>

      {/* Filter Dropdowns */}
      <AnimatePresence>
        {showFilters && (
          <motion.div
            initial={{ opacity: 0, height: 0 }}
            animate={{ opacity: 1, height: 'auto' }}
            exit={{ opacity: 0, height: 0 }}
            transition={{ duration: 0.2 }}
            className="overflow-hidden"
          >
            <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-4 p-4 bg-gray-50 rounded-xl border border-gray-200">
              {/* Event Type */}
              <div>
                <label className="block text-xs font-medium text-gray-500 mb-2 uppercase tracking-wider">
                  Event Type
                </label>
                <select
                  value={filters.eventType}
                  onChange={(e) => updateFilter('eventType', e.target.value as EventType | 'all')}
                  className="w-full px-3 py-2.5 bg-white border border-gray-200 rounded-lg 
                             text-gray-900 text-sm focus:outline-none focus:border-[#E11D2E] transition-colors"
                >
                  <option value="all">All Types</option>
                  {eventTypes.map((type) => (
                    <option key={type} value={type}>
                      {formatEventType(type)}
                    </option>
                  ))}
                </select>
              </div>

              {/* Country */}
              <div>
                <label className="block text-xs font-medium text-gray-500 mb-2 uppercase tracking-wider">
                  Country
                </label>
                <select
                  value={filters.country}
                  onChange={(e) => updateFilter('country', e.target.value)}
                  className="w-full px-3 py-2.5 bg-white border border-gray-200 rounded-lg 
                             text-gray-900 text-sm focus:outline-none focus:border-[#E11D2E] transition-colors"
                >
                  <option value="all">All Countries</option>
                  {countries.map((country) => (
                    <option key={country} value={country}>
                      {country}
                    </option>
                  ))}
                </select>
              </div>

              {/* Date Range */}
              <div>
                <label className="block text-xs font-medium text-gray-500 mb-2 uppercase tracking-wider">
                  Date Range
                </label>
                <select
                  value={filters.dateRange}
                  onChange={(e) => updateFilter('dateRange', e.target.value as EventFilters['dateRange'])}
                  className="w-full px-3 py-2.5 bg-white border border-gray-200 rounded-lg 
                             text-gray-900 text-sm focus:outline-none focus:border-[#E11D2E] transition-colors"
                >
                  <option value="all">All Dates</option>
                  <option value="this-week">This Week</option>
                  <option value="this-month">This Month</option>
                  <option value="next-month">Next Month</option>
                </select>
              </div>

              {/* Ticket Availability */}
              <div>
                <label className="block text-xs font-medium text-gray-500 mb-2 uppercase tracking-wider">
                  Tickets
                </label>
                <select
                  value={filters.ticketAvailable === null ? 'all' : filters.ticketAvailable ? 'available' : 'invite'}
                  onChange={(e) => {
                    const val = e.target.value;
                    updateFilter('ticketAvailable', val === 'all' ? null : val === 'available');
                  }}
                  className="w-full px-3 py-2.5 bg-white border border-gray-200 rounded-lg 
                             text-gray-900 text-sm focus:outline-none focus:border-[#E11D2E] transition-colors"
                >
                  <option value="all">All Events</option>
                  <option value="available">Tickets Available</option>
                  <option value="invite">Invite Only</option>
                </select>
              </div>
            </div>
          </motion.div>
        )}
      </AnimatePresence>
    </div>
  );
}
